/**
 * 
 */
package gov.va.med.mhv.usermgmt.integration.adapter;

import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.tigris.atlas.messages.MessageFactory;
import org.tigris.atlas.messages.Severity;
import org.tigris.atlas.service.VoidServiceResponse;

import gov.va.med.mhv.core.util.Precondition;
import gov.va.med.mhv.integration.phr.transfer.PatientExtractStatus;
import gov.va.med.mhv.integration.phr.transfer.PatientIdentifier;
import gov.va.med.mhv.integration.phr.transfer.RefreshDataParameters;
import gov.va.med.mhv.usermgmt.bizobj.ExtractControlBO;
import gov.va.med.mhv.usermgmt.enumeration.ExtractType;
import gov.va.med.mhv.usermgmt.integration.service.delegate.IntegrationServiceDelegateFactory;
import gov.va.med.mhv.usermgmt.integration.service.delegate.PatientDataServiceDelegate;
import gov.va.med.mhv.usermgmt.integration.service.util.PhrInterfaceUtils;
import gov.va.med.mhv.usermgmt.service.AccessControlCollectionServiceResponse;
import gov.va.med.mhv.usermgmt.service.FacilityExtractStatusCollectionServiceResponse;
import gov.va.med.mhv.usermgmt.service.adapter.PhrConfigProperties;
import gov.va.med.mhv.usermgmt.service.delegate.AccessControlServiceDelegate;
import gov.va.med.mhv.usermgmt.service.delegate.ServiceDelegateFactory;
import gov.va.med.mhv.usermgmt.service.handler.MviProperties;
import gov.va.med.mhv.usermgmt.transfer.Patient;
import gov.va.med.mhv.usermgmt.transfer.UserProfile;
import gov.va.med.mhv.usermgmt.util.AccessControlUtils;
import gov.va.med.mhv.usermgmt.util.AccessDomainUtils;
import gov.va.med.mhv.usermgmt.util.AccessPermissionUtils;
import gov.va.med.mhv.usermgmt.util.ExtractEnablementStatusUtils;

/**
 * @author Rob Proper (Aquilent Inc.)
 *
 */
public final class PatientExtractServiceAdapter {

    private static final Log LOG = LogFactory.getLog(
        PatientExtractServiceAdapter.class);

    private static boolean isAnyExtractEnabled() {
        List controls = ExtractControlBO.getEnabledExtracts();
        if (controls != null) {
            for(Object e: controls) {
                ExtractControlBO control = (ExtractControlBO) e;
                if ((control != null) && 
                		(control.getEnablementStatus() ==
                			ExtractEnablementStatusUtils.ENABLED  ||
                			control.getEnablementStatus() ==
                    			ExtractEnablementStatusUtils.PARTIAL))
                {
                    return true;
                }
            }
        }
        return false;
    }
    
     /**
     * 
     * @param patient
     * @param stationNumbers
     */
    public VoidServiceResponse updateExtract(Patient patient, 
        List<String> stationNumbers) 
    {
        Precondition.assertNotNull("patient", patient);
        Precondition.assertNotNull("patient.userProfile", 
            patient.getUserProfile());
        Precondition.assertNotNull("stationNumbers", stationNumbers);

        if (!isAnyExtractEnabled()) {
            LOG.info("Skipped updateExtract for " + getUserName(patient) + 
                " and station numbers " + stationNumbers + 
                ", because All extracts are disabled!");
            // TODO Add info message to response stating that all extracts 
            // are disabled
            return new VoidServiceResponse();
        }
        
        if (LOG.isDebugEnabled()) {
            LOG.debug("updateExtract(" + getUserName(patient)
                + ", " + stationNumbers + ")");
        }
        PhrConfigProperties phrProp = PhrConfigProperties.getInstance();
    
        // Adding for new PHR Refresh
        if (phrProp.isNewphr_enabled() && hasNewPhrAccess(patient.getUserProfile()))  {
	        PhrManagerAdapter phrManager = new PhrManagerAdapter();
	        try {
				phrManager.refreshData(patient, stationNumbers);
			} catch (Exception e) {
				e.printStackTrace();
			}
        }
        else {
        	long s = System.currentTimeMillis();
	        RefreshDataParameters parameters = new RefreshDataParameters(
	            new PatientIdentifier(patient.getIcn()));
	        getDelegate().refreshData(parameters);
	        if(LOG.isInfoEnabled()) {
	        	LOG.info("OLD Phr refreshData took " + (System.currentTimeMillis() -s) +" ms for UserProfile: " +patient.getUserProfile().getId());
	        }	
        }
        return new VoidServiceResponse();
    }

    /**
     * 
     * @param patient
     * @param stationNumbers
     */
    public FacilityExtractStatusCollectionServiceResponse getExtractStatus(
        ExtractType extractType, Patient patient) 
    {
        Precondition.assertNotNull("extractType", extractType);
        Precondition.assertNotBlank("extractType.name", extractType.getName());
        Precondition.assertNotNull("patient", patient);

        FacilityExtractStatusCollectionServiceResponse response = 
            new FacilityExtractStatusCollectionServiceResponse();

        if (!isAnyExtractEnabled()) {
            LOG.info("Skipped getExtractStatus for " + getUserName(patient) + 
                ", because all extracts are disabled!");
            // TODO Add info message to response stating that all extracts 
            // are disabled
            return response;
        }
        
        PhrConfigProperties phrProp = PhrConfigProperties.getInstance();
        PatientExtractStatus status = null;
    
        //Adding for New Phr Manager
        if( phrProp.isNewphr_enabled() && hasNewPhrAccess(patient.getUserProfile())) {
	        PhrManagerAdapter phrManager = new PhrManagerAdapter();
	        try {
	        	status = phrManager.getExtractStatus(extractType, patient);
			} catch (Exception e) {
				e.printStackTrace();
			}
        }
        else {
        	long s = System.currentTimeMillis();
	        status = getDelegate().getStatus(new PatientIdentifier(patient.getIcn()));
	        if(LOG.isInfoEnabled()) {
	        	LOG.info("OLD Phr Extract took " + (System.currentTimeMillis() -s) +" ms for UserProfile: " +patient.getUserProfile().getId());
	        }	
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("getExtractStatus for " + patient.getUserProfile().
                getUserName() + " yields " + PhrInterfaceUtils.describe(
                status));
        }
         if(status == null) {
	        org.tigris.atlas.messages.Message m = MessageFactory.createMessage();
	        m.setKey("phr.extract.down");
	        m.setSeverity(Severity.getSeverity("Error"));
	        response.getMessages().addMessage(m);
        }
        else {
        response.addItems(PhrInterfaceUtils.toFacilityExtractStatusses(patient, 
            extractType, status));
        }
        return response;
    }
    

    /**
     * 
     * @param patient The patient to delete the extract for. 
     * @param stationNumbers
     */
    public VoidServiceResponse deleteExtract(Patient patient, 
        List<String> stationNumbers) 
    {
        Precondition.assertNotNull("patient", patient);
        Precondition.assertNotNull("patient.icn", patient.getIcn());
        Precondition.assertNotNull("stationNumbers", stationNumbers);
        for (String stationNumber: stationNumbers) {
            getDelegate().removePhrData(new PatientIdentifier(patient.
                getIcn()), stationNumber);
        }
        return new VoidServiceResponse();
    }

    private PatientDataServiceDelegate getDelegate() {
        return IntegrationServiceDelegateFactory.
            createPatientDataServiceDelegate();
    }

    private String getUserName(Patient patient) {
        return ((patient != null) && (patient.getUserProfile() != null))
            ? "patient '" + patient.getUserProfile().getUserName() + "'"
            : "<unknown patient>";
    }

    private boolean hasNewPhrAccess(UserProfile userProfile)
        {
		boolean hasExtractAccess = false;
		AccessControlCollectionServiceResponse accessResponse =
			getAccessControlServiceDelegate().findAccessControls(userProfile.getUserName());
	 //   Precondition.assertNotNull("response", response);
		hasExtractAccess = AccessControlUtils.hasAccess(AccessDomainUtils.NEWPHR,
	    	"NEWPHR",AccessPermissionUtils.READ, accessResponse.getAccessControls()); // "unchecked"
		return hasExtractAccess;
	  }
	public AccessControlServiceDelegate getAccessControlServiceDelegate() {
		return ServiceDelegateFactory.createAccessControlServiceDelegate();
	}
    
}